#ifndef OSMIUM_UTIL_DELTA_HPP
#define OSMIUM_UTIL_DELTA_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2015 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <iterator>
#include <type_traits>
#include <utility>

namespace osmium {

    namespace util {

        /**
         * Helper class for delta encoding.
         */
        template <typename T>
        class DeltaEncode {

            T m_value;

        public:

            DeltaEncode(T value = 0) :
                m_value(value) {
            }

            void clear() {
                m_value = 0;
            }

            T update(T new_value) {
                using std::swap;
                swap(m_value, new_value);
                return m_value - new_value;
            }

        }; // class DeltaEncode

        /**
         * Helper class for delta decoding.
         */
        template <typename T>
        class DeltaDecode {

            T m_value;

        public:

            DeltaDecode() :
                m_value(0) {
            }

            void clear() {
                m_value = 0;
            }

            T update(T delta) {
                m_value += delta;
                return m_value;
            }

        }; // class DeltaDecode

        template <typename TBaseIterator, typename TTransform, typename TValue>
        class DeltaEncodeIterator : public std::iterator<std::input_iterator_tag, TValue> {

            typedef TValue value_type;

            TBaseIterator m_it;
            TBaseIterator m_end;
            value_type m_delta;
            DeltaEncode<value_type> m_value;
            TTransform m_trans;

        public:

            DeltaEncodeIterator(TBaseIterator first, TBaseIterator last, TTransform& trans) :
                m_it(first),
                m_end(last),
                m_delta(m_trans(m_it)),
                m_value(m_delta),
                m_trans(trans) {
            }

            DeltaEncodeIterator& operator++() {
                if (m_it != m_end) {
                    m_delta = m_value.update(m_trans(++m_it));
                }
                return *this;
            }

            DeltaEncodeIterator operator++(int) {
                DeltaEncodeIterator tmp(*this);
                operator++();
                return tmp;
            }

            value_type operator*() {
                return m_delta;
            }

            bool operator==(const DeltaEncodeIterator& rhs) const {
                return m_it == rhs.m_it && m_end == rhs.m_end;
            }

            bool operator!=(const DeltaEncodeIterator& rhs) const {
                return !(*this == rhs);
            }

        }; // class DeltaEncodeIterator

    } // namespace util

} // namespace osmium

#endif // OSMIUM_UTIL_DELTA_HPP
