#ifndef OSMIUM_OSM_TYPES_FROM_STRING_HPP
#define OSMIUM_OSM_TYPES_FROM_STRING_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2015 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <cassert>
#include <cctype>
#include <cstdint>
#include <cstdlib>
#include <limits>
#include <string>
#include <utility>

#include <osmium/osm/entity_bits.hpp>
#include <osmium/osm/types.hpp>

namespace osmium {

    inline object_id_type string_to_object_id(const char* input) {
        assert(input);
        if (*input != '\0' && !std::isspace(*input)) {
            char* end;
            auto id = std::strtoll(input, &end, 10);
            if (id != std::numeric_limits<long long>::min() && id != std::numeric_limits<long long>::max() && *end == '\0') {
                return id;
            }
        }
        throw std::range_error(std::string("illegal id: '") + input + "'");
    }

    inline std::pair<osmium::item_type, osmium::object_id_type> string_to_object_id(const char* input, osmium::osm_entity_bits::type types) {
        assert(input);
        assert(types != osmium::osm_entity_bits::nothing);
        if (*input != '\0') {
            if (std::isdigit(*input)) {
                return std::make_pair(osmium::item_type::undefined, string_to_object_id(input));
            }
            osmium::item_type t = osmium::char_to_item_type(*input);
            if (osmium::osm_entity_bits::from_item_type(t) & types) {
                return std::make_pair(t, string_to_object_id(input+1));
            }
        }
        throw std::range_error(std::string("not a valid id: '") + input + "'");
    }

    namespace detail {

        inline long string_to_ulong(const char* input, const char *name) {
            if (*input != '\0' && *input != '-' && !std::isspace(*input)) {
                char* end;
                auto value = std::strtoul(input, &end, 10);
                if (value != std::numeric_limits<unsigned long>::max() && *end == '\0') {
                    return value;
                }
            }
            throw std::range_error(std::string("illegal ") + name + ": '" + input + "'");
        }

    } // namespace detail

    inline object_version_type string_to_object_version(const char* input) {
        assert(input);
        return static_cast<object_version_type>(detail::string_to_ulong(input, "version"));
    }

    inline changeset_id_type string_to_changeset_id(const char* input) {
        assert(input);
        return static_cast<changeset_id_type>(detail::string_to_ulong(input, "changeset"));
    }

    inline signed_user_id_type string_to_user_id(const char* input) {
        assert(input);
        if (input[0] == '-' && input[1] == '1' && input[2] == '\0') {
            return -1;
        }
        return static_cast<signed_user_id_type>(detail::string_to_ulong(input, "user id"));
    }

    inline num_changes_type string_to_num_changes(const char* input) {
        assert(input);
        return static_cast<num_changes_type>(detail::string_to_ulong(input, "value for num changes"));
    }

} // namespace osmium

#endif // OSMIUM_OSM_TYPES_FROM_STRING_HPP
