/*
 * @(#)DirectoryChannelLoggerFactory.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.logger;

import java.io.File;
import java.util.Properties;

import net.sourceforge.groboutils.codecoverage.v2.IChannelLogger;
import net.sourceforge.groboutils.codecoverage.v2.IChannelLoggerFactory;


/**
 * The singleton invoked at runtime to log each marked bytecode instruction
 * covered.
 * <P>
 * This class needs to be fast, efficient, thread-safe, and classloader-safe.
 * "Classloader-safe" means that it needs to be resiliant to multiple instances
 * of this class being loaded, and possibly interfering with each other.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/05/04 06:40:09 $
 * @since     December 17, 2002
 */
public class DirectoryChannelLoggerFactory implements IChannelLoggerFactory
{
    public static final String DIRECTORY_PROPERTY = "dir";
    public static final String DEFAULT_DIRECTORY = "./.cover-logs";
    
    /**
     * Creates a specific logger type.  Initializes the logger based on the
     * given collection of properties.
     *
     * @param propertyPrefix the prefix that all logger properties will begin
     *    with.  Expect all logger-specific properties to be appended directly
     *    to this string.
     * @param props the property collection to pull the logger properties from.
     * @param channelIndex channel number to log to.
     * @return the initialized logger.
     */
    public IChannelLogger createChannelLogger(
            String propertyPrefix, Properties props, short channelIndex )
    {
        String directory = getDirectory( propertyPrefix, props );
        File dir = new File( directory, Short.toString( channelIndex ) );
        if (dir.exists())
        {
            if (!dir.isDirectory())
            {
                System.err.println(
                    "DirectoryLogger base directory is a file.");
                dir = null;
            }
        }
        else
        {
            dir.mkdirs();
        }
        return new DirectoryChannelLogger( dir );
    }
    
    
    protected String getDirectory( String propertyPrefix, Properties props )
    {
        String directory = props.getProperty(
            propertyPrefix + DIRECTORY_PROPERTY );
        if (directory == null)
        {
            directory = DEFAULT_DIRECTORY;
        }
        return directory;
    }
}

